<?php
/*
Plugin Name: Grade de Atalhos com SVG
Description: Plugin para exibir atalhos com ícones SVG, ordenados por cliques e agrupados por categorias.
Version: 2.0 - stable version
Author: Rafaths - contato@rafaths.com.br
*/

// Permitir SVG com verificação de permissões
add_filter('upload_mimes', function ($mimes) {
    if (current_user_can('manage_options')) {
        $mimes['svg'] = 'image/svg+xml';
    }
    return $mimes;
});

// Corrige erro ao validar tipo MIME do SVG
add_filter('wp_check_filetype_and_ext', function ($data, $file, $filename, $mimes) {
    $ext = pathinfo($filename, PATHINFO_EXTENSION);
    if ($ext === 'svg') {
        $data['ext'] = 'svg';
        $data['type'] = 'image/svg+xml';
    }
    return $data;
}, 10, 4);



// Registrar CPT e Taxonomia
add_action('init', function () {
    register_post_type('atalho', [
        'labels' => [
            'name' => 'Atalhos',
            'singular_name' => 'Atalho',
            'add_new_item' => 'Adicionar Novo Atalho'
        ],
        'public' => true,
        'menu_icon' => 'dashicons-screenoptions',
        'supports' => ['title'],
        'show_in_rest' => true
    ]);

    register_taxonomy('grupo_atalho', 'atalho', [
        'labels' => [
            'name' => 'Grupos de Atalhos',
            'singular_name' => 'Grupo de Atalho'
        ],
        'public' => true,
        'hierarchical' => true,
        'show_in_rest' => true
    ]);
});

add_action('admin_head', function () {
    $screen = get_current_screen();
    if ($screen->post_type === 'atalho') {
        echo '<style>#postcustom { display: none !important; }</style>';
    }
});

// 1. Adiciona a coluna "Grupo" antes da coluna "Data"
add_filter('manage_atalho_posts_columns', function ($columns) {
    $new_columns = [];
    foreach ($columns as $key => $value) {
        if ($key === 'date') {
            $new_columns['grupo_atalho'] = 'Grupo';
        }
        $new_columns[$key] = $value;
    }
    return $new_columns;
});

// 2. Preenche a coluna com o nome do grupo
add_action('manage_atalho_posts_custom_column', function ($column, $post_id) {
    if ($column === 'grupo_atalho') {
        $terms = get_the_terms($post_id, 'grupo_atalho');
        if ($terms && !is_wp_error($terms)) {
            echo esc_html(implode(', ', wp_list_pluck($terms, 'name')));
        } else {
            echo '<em>Sem grupo</em>';
        }
    }
}, 10, 2);

// 3. Torna a coluna "Grupo" ordenável
add_filter('manage_edit-atalho_sortable_columns', function ($columns) {
    $columns['grupo_atalho'] = 'grupo_atalho';
    return $columns;
});

// 4. Altera a query para permitir ordenação por grupo
add_action('pre_get_posts', function ($query) {
    if (!is_admin() || !$query->is_main_query()) return;

    if ($query->get('orderby') === 'grupo_atalho') {
        $query->set('orderby', 'taxonomy');
        $query->set('tax_query', [[
            'taxonomy' => 'grupo_atalho',
            'field' => 'slug',
            'terms' => get_terms(['taxonomy' => 'grupo_atalho', 'fields' => 'slugs']),
            'operator' => 'IN'
        ]]);
    }
});


// Adiciona o código hexa das cores dos grupos na frente do nome
add_filter('wp_terms_checklist_args', function ($args, $post_id) {
    if ($args['taxonomy'] !== 'grupo_atalho') return $args;

    $args['walker'] = new class extends Walker_Category_Checklist {
        function start_el(&$output, $term, $depth = 0, $args = [], $id = 0) {
            $term_id = $term->term_id;
            $cor = get_term_meta($term_id, '_ga_grupo_cor', true);

            $name = '';

            if ($cor) {
                $name .= '<span style="
                    display:inline-block;
                    width:16px;
                    height:16px;
                    background:' . esc_attr($cor) . ';
                    margin-right:6px;
                    vertical-align:middle;
                    border:1px solid #ccc;
                    "></span>';
            }

            $name .= '<strong>' . esc_html($term->name) . '</strong>';

            $input_id = "in-{$args['taxonomy']}-{$term_id}";
            $checked = in_array($term_id, (array)$args['selected_cats']) ? 'checked="checked"' : '';

            $color_style = $cor ? " style='color: {$cor};'" : "";

            $output .= "<li id='{$input_id}'>";
            $output .= "<label class='selectit'{$color_style}>";
            $output .= "<input value='{$term_id}' type='checkbox' name='tax_input[{$args['taxonomy']}][]' id='{$input_id}-2' {$checked}> ";
            $output .= $name;
            $output .= "</label></li>";
        }
    };

    return $args;
}, 10, 2);



// Meta boxes para os atalhos
add_action('add_meta_boxes', function () {
    add_meta_box('ga_dados_atalho', 'Detalhes do Atalho', function ($post) {
        $f = fn($k) => get_post_meta($post->ID, $k, true);
        ?>
        <label>Link:</label><input type="url" name="ga_link" value="<?= esc_attr($f('_ga_link')) ?>" style="width:100%"><br><br>
        <label>Descrição:</label><textarea name="ga_descricao" style="width:100%" rows="3"><?= esc_textarea($f('_ga_descricao')) ?></textarea><br><br>
        <label>Cor do Fundo:</label><input type="color" name="ga_cor_fundo" value="<?= esc_attr($f('_ga_cor_fundo')) ?>"><br><br>
        <label>Cor do Ícone:</label><input type="color" name="ga_cor_icone" value="<?= esc_attr($f('_ga_cor_icone') ?: '#ffffff') ?>"><br><br>
        <label>SVG:</label>
        <input type="text" name="ga_svg_url" id="ga_svg_url" value="<?= esc_attr($f('_ga_svg_url')) ?>" style="width:100%">
        <button type="button" class="button" id="select-svg-button">Selecionar SVG</button>
        <div id="svg-preview" style="margin-top:10px;">
            <?php if ($f('_ga_svg_url')) echo '<img src="' . esc_url($f('_ga_svg_url')) . '" style="max-width:40px">'; ?>
        </div><br><br>
        <label><input type="checkbox" name="ga_ativo" value="1" <?= checked($f('_ga_ativo'), '1', false) ?>> Atalho ativo?</label><br><br>
        <label>Cliques:</label><input type="text" value="<?= intval($f('_ga_cliques')) ?>" readonly>
        <?php
    }, 'atalho');
});

// Salvar metas do atalho
add_action('save_post', function ($id) {
    foreach (['ga_link', 'ga_descricao', 'ga_cor_fundo', 'ga_cor_icone', 'ga_svg_url'] as $key) {
        if (isset($_POST[$key])) update_post_meta($id, "_$key", sanitize_text_field($_POST[$key]));
    }
    update_post_meta($id, '_ga_ativo', isset($_POST['ga_ativo']) ? '1' : '0');
});

// Adicionar campos personalizados aos grupos
add_action('grupo_atalho_add_form_fields', function () {
    echo '<div class="form-field"><label for="ga_grupo_cor">Cor do Grupo</label><input type="color" name="ga_grupo_cor" value="#000000"></div>';
    echo '<div class="form-field"><label><input type="checkbox" name="ga_grupo_ativo" value="1" checked> Grupo ativo?</label></div>';
});

add_action('grupo_atalho_edit_form_fields', function ($term) {
    $cor = get_term_meta($term->term_id, '_ga_grupo_cor', true) ?: '#000000';
    $ativo = get_term_meta($term->term_id, '_ga_grupo_ativo', true);
    echo "<tr class='form-field'><th><label for='ga_grupo_cor'>Cor do Grupo</label></th><td><input type='color' name='ga_grupo_cor' value='" . esc_attr($cor) . "'></td></tr>";
    echo "<tr class='form-field'><th><label for='ga_grupo_ativo'>Grupo ativo?</label></th><td><input type='checkbox' name='ga_grupo_ativo' value='1' " . checked($ativo, '1', false) . "></td></tr>";
});

add_action('edited_grupo_atalho', 'ga_save_grupo_meta');
add_action('created_grupo_atalho', 'ga_save_grupo_meta');
function ga_save_grupo_meta($term_id) {
    update_term_meta($term_id, '_ga_grupo_cor', sanitize_text_field($_POST['ga_grupo_cor'] ?? ''));
    update_term_meta($term_id, '_ga_grupo_ativo', isset($_POST['ga_grupo_ativo']) ? '1' : '0');
}

// Clique tracker
add_action('wp_ajax_ga_click', 'ga_click');
add_action('wp_ajax_nopriv_ga_click', 'ga_click');
function ga_click() {
    $id = intval($_POST['id']);
    $c = intval(get_post_meta($id, '_ga_cliques', true));
    update_post_meta($id, '_ga_cliques', $c + 1);
    wp_send_json_success();
}

// Shortcode
add_shortcode('grade_atalhos', function () {
    ob_start();
    
    echo '<div class="ga-pesquisa"><input type="text" id="ga-search" placeholder="Pesquisar serviços"></div>';
    echo '<div id="ga-resultados" class="ga-grid ga-resultados"></div>';

    echo '<h2>Mais Acessados</h2><div class="ga-grid">';
    $q = new WP_Query([
        'post_type' => 'atalho',
        'posts_per_page' => 9,
        'meta_key' => '_ga_cliques',
        'orderby' => 'meta_value_num',
        'order' => 'DESC',
        'meta_query' => [['key' => '_ga_ativo', 'value' => '1']]
    ]);
    while ($q->have_posts()) {
        $q->the_post();
        $terms = get_the_terms(get_the_ID(), 'grupo_atalho');
        if (!$terms) continue;
        $grupo = $terms[0];
        if (get_term_meta($grupo->term_id, '_ga_grupo_ativo', true) !== '1') continue;
        ga_render_atalho(get_the_ID());
    }
    wp_reset_postdata();
    echo '</div>';

    foreach (get_terms(['taxonomy' => 'grupo_atalho', 'hide_empty' => false]) as $g) {
        if (get_term_meta($g->term_id, '_ga_grupo_ativo', true) !== '1') continue;

        $qq = new WP_Query([
            'post_type' => 'atalho',
            'tax_query' => [[ 'taxonomy' => 'grupo_atalho', 'terms' => $g->term_id ]],
            'meta_query' => [[ 'key' => '_ga_ativo', 'value' => '1' ]]
        ]);

        if (!$qq->have_posts()) continue;

        echo '<h2 style="color:' . esc_attr(get_term_meta($g->term_id, '_ga_grupo_cor', true)) . '">' . esc_html($g->name) . '</h2><div class="ga-grid">';
        while ($qq->have_posts()) {
            $qq->the_post();
            ga_render_atalho(get_the_ID());
        }
        echo '</div>';
        wp_reset_postdata();
    }

    return ob_get_clean();
});

// Renderização dos atalhos
function ga_render_atalho($id) {
    $f = fn($k) => get_post_meta($id, $k, true);
    $svg = '';

    if ($svg_url = $f('_ga_svg_url')) {
        $svg_path = str_replace(site_url('/'), ABSPATH, $svg_url);
        if (file_exists($svg_path)) {
            $svg_content = file_get_contents($svg_path);
            $color = esc_attr($f('_ga_cor_icone'));
            $svg = preg_replace('/<svg([^>]*)>/i', '<svg$1 class="ga-icon-svg" fill="' . $color . '">', $svg_content);
        }
    }

    echo '<a href="' . esc_url($f('_ga_link')) . '" class="ga-atalho" data-id="' . $id . '" target="_blank" rel="noopener noreferrer">
        <div class="ga-icon" style="background-color:' . esc_attr($f('_ga_cor_fundo')) . '">'
            . $svg .
        '</div>
        <div><strong>' . esc_html(get_the_title($id)) . '</strong><br><small>' . esc_html($f('_ga_descricao')) . '</small></div>
    </a>';
}

// Scripts admin
add_action('admin_enqueue_scripts', function ($hook) {
    if ($hook !== 'post.php' && $hook !== 'post-new.php') return;
    wp_enqueue_media();
    wp_enqueue_script('svg-upload', plugin_dir_url(__FILE__) . 'js/svg-upload.js', ['jquery'], null, true);
});

// Scripts
add_action('wp_enqueue_scripts', function () {
    wp_enqueue_style('grade-atalhos-style', plugin_dir_url(__FILE__) . 'css/grade-atalhos.css');
    wp_enqueue_script('grade-atalhos-script', plugin_dir_url(__FILE__) . 'js/grade-atalhos.js', [], null, true);
    wp_localize_script('grade-atalhos-script', 'ga_vars', ['ajax_url' => admin_url('admin-ajax.php')]);
});