<?php
/*
Plugin Name: RTS Grade de Atalhos com SVG
Description: Plugin para exibir atalhos com ícones SVG, ordenados por cliques e agrupados por categorias.
Version: 4.1
Author: Rafaths - contato@rafaths.com.br
*/

// Permitir SVG com verificação de permissões
add_filter('upload_mimes', function ($mimes) {
    if (current_user_can('manage_options')) {
        $mimes['svg'] = 'image/svg+xml';
    }
    return $mimes;
});

// Corrige erro ao validar tipo MIME do SVG
add_filter('wp_check_filetype_and_ext', function ($data, $file, $filename, $mimes) {
    $ext = pathinfo($filename, PATHINFO_EXTENSION);
    if ($ext === 'svg') {
        $data['ext'] = 'svg';
        $data['type'] = 'image/svg+xml';
    }
    return $data;
}, 10, 4);



add_action('admin_menu', function () {
    // Remove o submenu "Adicionar Novo" duplicado, se quiser
    remove_submenu_page('edit.php?post_type=atalho', 'post-new.php?post_type=atalho');

    // Adiciona o submenu "Grupos de Atalhos" (taxonomia)
    add_submenu_page(
        'edit.php?post_type=atalho',
        'Grupos de Atalhos',
        'Grupos de Atalhos',
        'manage_options',
        'edit-tags.php?taxonomy=grupo_atalho&post_type=atalho'
    );

    // Adiciona o submenu "Personalizar Grade" (configurações)
    add_submenu_page(
        'edit.php?post_type=atalho',
        'Personalizar Estilo da Grade',
        'Personalizar Grade',
        'manage_options',
        'grade-atalhos-settings',
        'render_grade_atalhos_settings_page'
    );
});


// Adiciona coluna "Status" aos atalhos
add_filter('manage_atalho_posts_columns', function($columns) {
    $columns['ga_status'] = 'Status';
    return $columns;
});

add_action('manage_atalho_posts_custom_column', function($column, $post_id) {
    if ($column === 'ga_status') {
        $ativo = get_post_meta($post_id, '_ga_ativo', true);
        echo $ativo === '1' ? '<span style="color:green;font-weight:bold;">Ativo</span>' : '<span style="color:red;">Inativo</span>';
    }
}, 10, 2);


// Adiciona coluna "Status" na lista de termos da taxonomia grupo_atalho
add_filter('manage_edit-grupo_atalho_columns', function($columns) {
    $columns['ga_status'] = 'Status';
    return $columns;
});

add_filter('manage_grupo_atalho_custom_column', function($content, $column_name, $term_id) {
    if ($column_name === 'ga_status') {
        $ativo = get_term_meta($term_id, '_ga_grupo_ativo', true);
        return $ativo === '1' ? '<span style="color:green;font-weight:bold;">Ativo</span>' : '<span style="color:red;">Inativo</span>';
    }
    return $content;
}, 10, 3);




function render_grade_atalhos_settings_page() {
    ?>
    <div class="wrap">
	
		<style>
			.ga-settings-wrapper h1,
			.ga-settings-wrapper h2,
			.ga-settings-wrapper h3,
			.ga-settings-wrapper h4,
			.ga-settings-wrapper h5,
			.ga-settings-wrapper h6 {
				margin: 0;
			}
		</style>
	
        <h1>Personalizar Estilo da Grade</h1>

        <div class="ga-settings-wrapper" style="display: flex; gap: 40px; align-items: flex-start;">
            <!-- Coluna do formulário -->
            <div class="ga-settings-form" style="flex: 1;">
                <form method="post" action="options.php">
                    <?php
                    settings_fields('grade_atalhos_settings');
                    do_settings_sections('grade_atalhos_settings');
                    ?>

                    <table class="form-table">
                        <tr>
                            <th>Cor de fundo do atalho</th>
                            <td><input type="color" name="ga_bg_color" id="ga_bg_color" value="<?php echo esc_attr(get_option('ga_bg_color', '#f5f5f5')); ?>"></td>
                        </tr>                
                        <tr>
                            <th>Cor da borda</th>
                            <td><input type="color" name="ga_border_color" id="ga_border_color" value="<?php echo esc_attr(get_option('ga_border_color', '#9ca3af')); ?>"></td>
                        </tr>
                        <tr>
                            <th>Cor da fonte</th>
                            <td><input type="text" class="color-picker-alpha" name="ga_font_color" id="ga_font_color" value="<?php echo esc_attr(get_option('ga_font_color', '#1E73BE')); ?>"></td>
                        </tr>
                        <tr>
                            <th>Raio da borda (px)</th>
                            <td><input type="range" name="ga_border_radius" id="ga_border_radius" min="0" max="50" value="<?php echo esc_attr(get_option('ga_border_radius', 8)); ?>"></td>
                        </tr>
                        <tr><th colspan="2"><h3>Box Shadow</h3></th></tr>
                        <tr>
                            <th>Deslocamento horizontal (X)</th>
                            <td><input type="range" name="ga_shadow_x" id="ga_shadow_x" min="-50" max="50" value="<?php echo esc_attr(get_option('ga_shadow_x', 0)); ?>"> <span id="ga_shadow_x_val">0px</span></td>
                        </tr>
                        <tr>
                            <th>Deslocamento vertical (Y)</th>
                            <td><input type="range" name="ga_shadow_y" id="ga_shadow_y" min="-50" max="50" value="<?php echo esc_attr(get_option('ga_shadow_y', 5)); ?>"> <span id="ga_shadow_y_val">5px</span></td>
                        </tr>
                        <tr>
                            <th>Desfoque (blur)</th>
                            <td><input type="range" name="ga_shadow_blur" id="ga_shadow_blur" min="0" max="100" value="<?php echo esc_attr(get_option('ga_shadow_blur', 10)); ?>"> <span id="ga_shadow_blur_val">10px</span></td>
                        </tr>
                        <tr>
                            <th>Spread</th>
                            <td><input type="range" name="ga_shadow_spread" id="ga_shadow_spread" min="-50" max="50" value="<?php echo esc_attr(get_option('ga_shadow_spread', 0)); ?>"> <span id="ga_shadow_spread_val">0px</span></td>
                        </tr>
                        <tr>
                            <th>Cor da sombra</th>
                            <td><input type="text" class="color-picker-alpha" name="ga_shadow_color" id="ga_shadow_color" value="<?php echo esc_attr(get_option('ga_shadow_color', 'rgba(0,0,0,0.35)')); ?>"></td>
                        </tr>
                        <tr>
                            <th>Atalhos por linha</th>
                            <td>
                                <select name="ga_columns" id="ga_columns">
                                    <?php
                                    $current = intval(get_option('ga_columns', 3));
                                    for ($i = 1; $i <= 6; $i++) {
                                        echo "<option value='{$i}'" . selected($current, $i, false) . ">{$i} por linha</option>";
                                    }
                                    ?>
                                </select>
                            </td>
                        </tr>
                        <tr>
                            <th>Tamanho do título</th>
                            <td>
                                <select name="ga_heading_tag" id="ga_heading_tag">
                                    <?php
                                    $current_tag = get_option('ga_heading_tag', 'h2');
                                    foreach (['h1', 'h2', 'h3', 'h4', 'h5', 'h6'] as $tag) {
                                        echo "<option value='{$tag}'" . selected($current_tag, $tag, false) . ">" . strtoupper($tag) . "</option>";
                                    }
                                    ?>
                                </select>
                            </td>
                        </tr>
                    </table>

                    <?php submit_button(); ?>
                </form>
            </div>

            <!-- Coluna do preview -->
            <div class="ga-preview-box" style="flex: 1; position: sticky; top: 100px;">		
				<h2>Pré-visualização</h2>
				<div id="ga-preview" style="display:inline-block; padding:20px; border:1px solid #ccc; margin-top:10px;">
					<div class="ga-atalho" style="display:flex; align-items:center; gap:10px; padding:10px;">
						<div class="ga-icon" style="width:80px; height:80px; display:flex; align-items:center; justify-content:center;background-color: #0075ff;border-radius: 10px;">
							<svg width="80" height="80"><circle cx="40" cy="40" r="28" fill="#ffffff" /></svg>
						</div>
						<div>
							<strong>Exemplo de Atalho</strong><br>
							<small>Descrição de exemplo</small>
						</div>
					</div>
				</div>
			</div>
        </div>
    </div>

    <script>
    document.addEventListener('DOMContentLoaded', function () {
        const bg = document.getElementById('ga_bg_color');
        const radius = document.getElementById('ga_border_radius');
        const border = document.getElementById('ga_border_color');
        const fontColor = document.getElementById('ga_font_color');
        const atalho = document.querySelector('#ga-preview .ga-atalho');
        const sx = document.getElementById('ga_shadow_x');
        const sy = document.getElementById('ga_shadow_y');
        const blur = document.getElementById('ga_shadow_blur');
        const spread = document.getElementById('ga_shadow_spread');
        const scolor = document.getElementById('ga_shadow_color');

        const sx_val = document.getElementById('ga_shadow_x_val');
        const sy_val = document.getElementById('ga_shadow_y_val');
        const blur_val = document.getElementById('ga_shadow_blur_val');
        const spread_val = document.getElementById('ga_shadow_spread_val');

        const grid = document.querySelector('#ga-preview');
        const columnsSelect = document.getElementById('ga_columns');

        function updatePreview() {
            atalho.style.backgroundColor = bg.value;
            atalho.style.borderRadius = radius.value + 'px';
            atalho.style.border = '1px solid ' + border.value;
            atalho.style.color = fontColor.value;
            atalho.style.boxShadow = `${sx.value}px ${sy.value}px ${blur.value}px ${spread.value}px ${scolor.value}`;
            grid.style.gridTemplateColumns = `repeat(${columnsSelect.value}, 1fr)`;

            sx_val.textContent = sx.value + 'px';
            sy_val.textContent = sy.value + 'px';
            blur_val.textContent = blur.value + 'px';
            spread_val.textContent = spread.value + 'px';
        }

        [bg, radius, border, fontColor, sx, sy, blur, spread, scolor, columnsSelect].forEach(el => {
            el.addEventListener('input', updatePreview);
        });

        updatePreview();
    });
    </script>
    <?php
}





// Registrar CPT e Taxonomia
add_action('init', function () {
    register_post_type('atalho', [
        'labels' => [
            'name' => 'Atalhos',
            'singular_name' => 'Atalho',
            'add_new_item' => 'Adicionar Novo Atalho'
        ],
        'public' => true,
        'menu_icon' => 'dashicons-screenoptions',
        'supports' => ['title'],
        'show_in_rest' => true
    ]);

    register_taxonomy('grupo_atalho', 'atalho', [
		'labels' => [
			'name' => 'Grupos de Atalhos',
			'singular_name' => 'Grupo de Atalho'
		],
		'public' => true,
		'hierarchical' => true,
		'show_in_rest' => true,
		'show_in_menu' => false, // evita menu automático duplicado
	]);

});

add_action('admin_head', function () {
    $screen = get_current_screen();
    if ($screen->post_type === 'atalho') {
        echo '<style>#postcustom { display: none !important; }</style>';
    }
});

// 1. Adiciona a coluna "Grupo" antes da coluna "Data"
add_filter('manage_atalho_posts_columns', function ($columns) {
    $new_columns = [];
    foreach ($columns as $key => $value) {
        if ($key === 'date') {
            $new_columns['grupo_atalho'] = 'Grupo';
        }
        $new_columns[$key] = $value;
    }
    return $new_columns;
});

// 2. Preenche a coluna com o nome do grupo
add_action('manage_atalho_posts_custom_column', function ($column, $post_id) {
    if ($column === 'grupo_atalho') {
        $terms = get_the_terms($post_id, 'grupo_atalho');
        if ($terms && !is_wp_error($terms)) {
            echo esc_html(implode(', ', wp_list_pluck($terms, 'name')));
        } else {
            echo '<em>Sem grupo</em>';
        }
    }
}, 10, 2);

// 3. Torna a coluna "Grupo" ordenável
add_filter('manage_edit-atalho_sortable_columns', function ($columns) {
    $columns['grupo_atalho'] = 'grupo_atalho';
    return $columns;
});

// 4. Altera a query para permitir ordenação por grupo
add_action('pre_get_posts', function ($query) {
    if (!is_admin() || !$query->is_main_query()) return;

    if ($query->get('orderby') === 'grupo_atalho') {
        $query->set('orderby', 'taxonomy');
        $query->set('tax_query', [[
            'taxonomy' => 'grupo_atalho',
            'field' => 'slug',
            'terms' => get_terms(['taxonomy' => 'grupo_atalho', 'fields' => 'slugs']),
            'operator' => 'IN'
        ]]);
    }
});


// Adiciona o código hexa das cores dos grupos na frente do nome
add_filter('wp_terms_checklist_args', function ($args, $post_id) {
    if ($args['taxonomy'] !== 'grupo_atalho') return $args;

    $args['walker'] = new class extends Walker_Category_Checklist {
        function start_el(&$output, $term, $depth = 0, $args = [], $id = 0) {
            $term_id = $term->term_id;
            $cor = get_term_meta($term_id, '_ga_grupo_cor', true);

            $name = '';

            if ($cor) {
                $name .= '<span style="
                    display:inline-block;
                    width:16px;
                    height:16px;
                    background:' . esc_attr($cor) . ';
                    margin-right:6px;
                    vertical-align:middle;
                    border:1px solid #ccc;
                    "></span>';
            }

            $name .= '<strong>' . esc_html($term->name) . '</strong>';

            $input_id = "in-{$args['taxonomy']}-{$term_id}";
            $checked = in_array($term_id, (array)$args['selected_cats']) ? 'checked="checked"' : '';

            $color_style = $cor ? " style='color: {$cor};'" : "";

            $output .= "<li id='{$input_id}'>";
            $output .= "<label class='selectit'{$color_style}>";
            $output .= "<input value='{$term_id}' type='checkbox' name='tax_input[{$args['taxonomy']}][]' id='{$input_id}-2' {$checked}> ";
            $output .= $name;
            $output .= "</label></li>";
        }
    };

    return $args;
}, 10, 2);



// Meta boxes para os atalhos
add_action('add_meta_boxes', function () {
    add_meta_box('ga_dados_atalho', 'Detalhes do Atalho', function ($post) {
        $f = fn($k) => get_post_meta($post->ID, $k, true);
        ?>
        <label>Descrição:</label><textarea name="ga_descricao" style="width:100%" rows="3"><?= esc_textarea($f('_ga_descricao')) ?></textarea><br><br>
        <label>Link:</label><input type="url" name="ga_link" value="<?= esc_attr($f('_ga_link')) ?>" style="width:100%"><br><br>
        <label>Cor do Fundo:</label><input type="color" name="ga_cor_fundo" value="<?= esc_attr($f('_ga_cor_fundo')) ?>"><br><br>
        <label>Cor do Ícone:</label><input type="color" name="ga_cor_icone" value="<?= esc_attr($f('_ga_cor_icone') ?: '#ffffff') ?>"><br><br>
        <label>SVG:</label>
        <input type="text" name="ga_svg_url" id="ga_svg_url" value="<?= esc_attr($f('_ga_svg_url')) ?>" style="width:100%">
        <button type="button" class="button" id="select-svg-button">Selecionar SVG</button>
        <div id="svg-preview" style="margin-top:10px;">
            <?php if ($f('_ga_svg_url')) echo '<img src="' . esc_url($f('_ga_svg_url')) . '" style="max-width:40px">'; ?>
        </div><br><br>
        <label><input type="checkbox" name="ga_ativo" value="1" <?= checked($f('_ga_ativo'), '1', false) ?>> Atalho ativo?</label><br><br>
        <label>Cliques:</label><input type="text" value="<?= intval($f('_ga_cliques')) ?>" readonly>
        <?php
    }, 'atalho');
});

// Salvar metas do atalho
add_action('save_post', function ($id) {
    foreach (['ga_link', 'ga_descricao', 'ga_cor_fundo', 'ga_cor_icone', 'ga_svg_url'] as $key) {
        if (isset($_POST[$key])) update_post_meta($id, "_$key", sanitize_text_field($_POST[$key]));
    }
    update_post_meta($id, '_ga_ativo', isset($_POST['ga_ativo']) ? '1' : '0');
});

// Adicionar campos personalizados aos grupos
add_action('grupo_atalho_add_form_fields', function () {
    echo '<div class="form-field"><label for="ga_grupo_cor">Cor do Grupo</label><input type="color" name="ga_grupo_cor" value="#000000"></div>';
    echo '<div class="form-field"><label><input type="checkbox" name="ga_grupo_ativo" value="1" checked> Grupo ativo?</label></div>';
});

add_action('grupo_atalho_edit_form_fields', function ($term) {
    $cor = get_term_meta($term->term_id, '_ga_grupo_cor', true) ?: '#000000';
    $ativo = get_term_meta($term->term_id, '_ga_grupo_ativo', true);
    echo "<tr class='form-field'><th><label for='ga_grupo_cor'>Cor do Grupo</label></th><td><input type='color' name='ga_grupo_cor' value='" . esc_attr($cor) . "'></td></tr>";
    echo "<tr class='form-field'><th><label for='ga_grupo_ativo'>Grupo ativo?</label></th><td><input type='checkbox' name='ga_grupo_ativo' value='1' " . checked($ativo, '1', false) . "></td></tr>";
});

add_action('edited_grupo_atalho', 'ga_save_grupo_meta');
add_action('created_grupo_atalho', 'ga_save_grupo_meta');
function ga_save_grupo_meta($term_id) {
    update_term_meta($term_id, '_ga_grupo_cor', sanitize_text_field($_POST['ga_grupo_cor'] ?? ''));
    update_term_meta($term_id, '_ga_grupo_ativo', isset($_POST['ga_grupo_ativo']) ? '1' : '0');
}

// Clique tracker
add_action('wp_ajax_ga_click', 'ga_click');
add_action('wp_ajax_nopriv_ga_click', 'ga_click');
function ga_click() {
    $id = intval($_POST['id']);
    $c = intval(get_post_meta($id, '_ga_cliques', true));
    update_post_meta($id, '_ga_cliques', $c + 1);
    wp_send_json_success();
}

// Shortcode
add_shortcode('grade_atalhos', function () {
    ob_start();

    echo '<div class="ga-pesquisa"><input type="text" id="ga-search" placeholder="Pesquisar serviços"></div>';
    echo '<div id="ga-resultados" class="ga-grid ga-resultados"></div>';

    // Mais Acessados
    $mais_acessados_cor = '#1e73be';
    echo "<fieldset style='
            border: 2px solid {$mais_acessados_cor};
            border-radius: 10px;
            margin-bottom: 20px;
            padding: 15px;
            box-shadow: 0 4px 8px rgba(0,0,0,0.2); /* Sombra */
        '>
        <legend style='
            color: {$mais_acessados_cor};
            font-weight: bold;
            font-size: 1.4em;
            padding: 0 8px; /* Espaço para o texto não encostar na borda */
        '>Mais Acessados</legend>";

    echo "<div class='ga-grid'>";

    $q = new WP_Query([
        'post_type'      => 'atalho',
        'posts_per_page' => 9,
        'meta_key'       => '_ga_cliques',
        'orderby'        => 'meta_value_num',
        'order'          => 'DESC',
        'meta_query'     => [['key' => '_ga_ativo', 'value' => '1']]
    ]);

    while ($q->have_posts()) {
        $q->the_post();
        $terms = get_the_terms(get_the_ID(), 'grupo_atalho');
        if (!$terms) continue;
        $grupo = $terms[0];
        if (get_term_meta($grupo->term_id, '_ga_grupo_ativo', true) !== '1') continue;
        ga_render_atalho(get_the_ID());
    }
    wp_reset_postdata();
    echo "</div></fieldset>";

    // Grupos
    foreach (get_terms(['taxonomy' => 'grupo_atalho', 'hide_empty' => false, 'orderby' => 'name', 'order' => 'ASC']) as $g) {
        if (get_term_meta($g->term_id, '_ga_grupo_ativo', true) !== '1') continue;

        $qq = new WP_Query([
            'post_type'      => 'atalho',
            'tax_query'      => [[ 'taxonomy' => 'grupo_atalho', 'terms' => $g->term_id ]],
            'meta_query'     => [[ 'key' => '_ga_ativo', 'value' => '1' ]],
            'orderby'        => 'title',  // Ordena pelo título (alfabética)
            'order'          => 'ASC'     // Ordem crescente (A-Z)
        ]);

        if (!$qq->have_posts()) continue;

        $cor_grupo = esc_attr(get_term_meta($g->term_id, '_ga_grupo_cor', true));
        if (!$cor_grupo) $cor_grupo = '#333'; // fallback se não houver cor

        echo "<fieldset style='
                border: 2px solid {$cor_grupo};
                border-radius: 10px;
                margin-bottom: 20px;
                padding: 15px;
                box-shadow: 0 4px 8px rgba(0,0,0,0.2); /* Sombra */
            '>
            <legend style='
                color: {$cor_grupo};
                font-weight: bold;
                font-size: 1.4em;
                padding: 0 8px;
            '>" . esc_html($g->name) . "</legend>";

        echo "<div class='ga-grid'>";
        while ($qq->have_posts()) {
            $qq->the_post();
            ga_render_atalho(get_the_ID());
        }
        echo "</div></fieldset>";

        wp_reset_postdata();
    }

    return ob_get_clean();
});




// Função para escurecer a cor
function darken_color($hex, $percent) {
    $hex = str_replace('#', '', $hex);
    if(strlen($hex) == 3) {
        $hex = $hex[0].$hex[0].$hex[1].$hex[1].$hex[2].$hex[2];
    }
    $r = hexdec(substr($hex, 0, 2));
    $g = hexdec(substr($hex, 2, 2));
    $b = hexdec(substr($hex, 4, 2));

    $r = max(0, min(255, $r - ($r * $percent / 100)));
    $g = max(0, min(255, $g - ($g * $percent / 100)));
    $b = max(0, min(255, $b - ($b * $percent / 100)));

    return sprintf("#%02x%02x%02x", $r, $g, $b);
}





// Renderização dos atalhos
function ga_render_atalho($id) {
    $f = fn($k) => get_post_meta($id, $k, true);
    $svg = '';

    if ($svg_url = $f('_ga_svg_url')) {
        $svg_path = str_replace(site_url('/'), ABSPATH, $svg_url);
        if (file_exists($svg_path)) {
            $svg_content = file_get_contents($svg_path);
            $color = esc_attr($f('_ga_cor_icone'));
            $svg = preg_replace('/<svg([^>]*)>/i', '<svg$1 class="ga-icon-svg" fill="' . $color . '">', $svg_content);
        }
    }

    echo '<a href="' . esc_url($f('_ga_link')) . '" class="ga-atalho" data-id="' . $id . '" target="_blank" rel="noopener noreferrer">
        <div class="ga-icon" style="background-color:' . esc_attr($f('_ga_cor_fundo')) . '">'
            . $svg .
        '</div>
        <div><strong>' . esc_html(get_the_title($id)) . '</strong><br><small>' . esc_html($f('_ga_descricao')) . '</small></div>
    </a>';
}

add_action('admin_init', function () {
    register_setting('grade_atalhos_settings', 'ga_bg_color');
    register_setting('grade_atalhos_settings', 'ga_border_radius');
    register_setting('grade_atalhos_settings', 'ga_border_color');
	register_setting('grade_atalhos_settings', 'ga_box_shadow');
	register_setting('grade_atalhos_settings', 'ga_min_width');
	register_setting('grade_atalhos_settings', 'ga_columns');
	register_setting('grade_atalhos_settings', 'ga_font_color');
	register_setting('grade_atalhos_settings', 'ga_heading_tag');
	foreach ([
		'ga_shadow_x', 'ga_shadow_y',
		'ga_shadow_blur', 'ga_shadow_spread',
		'ga_shadow_color'
	] as $opt) {
		register_setting('grade_atalhos_settings', $opt);
	}
});

add_action('wp_head', function () {
    $bg = get_option('ga_bg_color', '#f5f5f5');
    $radius = get_option('ga_border_radius', 8);
    $border = get_option('ga_border_color', '#9ca3af');
    $x = get_option('ga_shadow_x', 0);
    $y = get_option('ga_shadow_y', 5);
    $blur = get_option('ga_shadow_blur', 10);
    $spread = get_option('ga_shadow_spread', 0);
    $color = get_option('ga_shadow_color', '#000000');
    $columns = max(1, intval(get_option('ga_columns', 3))); // <-- ESSA LINHA FALTAVA
	$font_color = get_option('ga_font_color', '#1E73BE');

    echo "<style>
    .ga-atalho {
        background-color: {$bg};
        border-radius: {$radius}px;
        border: 1px solid {$border};
        box-shadow: {$x}px {$y}px {$blur}px {$spread}px {$color};
		color: {$font_color};
    }
    .ga-grid {
        display: grid;
        grid-template-columns: repeat($columns, 1fr);
        gap: 20px;
        margin-bottom: 40px;
    }
    </style>";
});




// Scripts admin
add_action('admin_enqueue_scripts', function ($hook) {
    if (isset($_GET['page']) && $_GET['page'] === 'grade-atalhos-settings') {
        wp_enqueue_style('wp-color-picker');
        wp_enqueue_script('wp-color-picker-alpha', plugin_dir_url(__FILE__) . 'js/wp-color-picker-alpha.js', ['wp-color-picker', 'jquery'], null, true);

        // Inicializar o color picker com alpha
        wp_add_inline_script('wp-color-picker-alpha', "
            jQuery(function($){
                $('.color-picker-alpha').wpColorPicker();
            });
        ");
    }

    // Inclui o script de seleção de SVGs no editor do post type "atalho"
    if ($hook === 'post.php' || $hook === 'post-new.php') {
        global $post;
        if (isset($post) && $post->post_type === 'atalho') {
            wp_enqueue_media();
            wp_enqueue_script('svg-upload', plugin_dir_url(__FILE__) . 'js/svg-upload.js', ['jquery'], null, true);
        }
    }
});


add_action('wp_enqueue_scripts', function () {
    wp_enqueue_style('grade-atalhos-style', plugin_dir_url(__FILE__) . 'css/grade-atalhos.css');
    wp_enqueue_script('grade-atalhos-script', plugin_dir_url(__FILE__) . 'js/grade-atalhos.js', ['jquery'], null, true);
    wp_localize_script('grade-atalhos-script', 'ga_vars', ['ajax_url' => admin_url('admin-ajax.php')]);
});