<?php
/*
Plugin Name: RTS Controle Avançado de Acesso a Páginas
Description: Permite criar funções personalizadas, atribuir permissões a páginas e restringir acesso para usuários.
Version: 2.0
Author: Rafaths - contato@rafaths.com.br
*/

// ✅ Adiciona menu no admin para gerenciar funções
add_action('admin_menu', function () {
    add_menu_page(
        'Controle de Acesso',
        'Controle de Acesso',
        'manage_options',
        'controle_acesso_paginas',
        'controle_acesso_paginas_admin',
        'dashicons-lock',
        81
    );
});

// ✅ Tela de administração de funções
function controle_acesso_paginas_admin() {
    if (!current_user_can('manage_options')) {
        wp_die(__('Você não tem permissão para acessar esta página.'));
    }

    // 📝 Cria função
    if (isset($_POST['nova_role']) && !empty($_POST['nova_role'])) {
        $role_name = sanitize_text_field($_POST['nova_role']);
        if (!get_role($role_name)) {
            add_role($role_name, ucfirst($role_name), [
                'read' => true,
                'edit_pages' => true,
                'edit_published_pages' => true,
                'edit_others_pages' => true,
                'publish_pages' => true,
                'delete_pages' => false,
            ]);
            echo '<div class="notice notice-success"><p>Função criada com sucesso!</p></div>';
        } else {
            echo '<div class="notice notice-error"><p>Essa função já existe.</p></div>';
        }
    }

    // 🗑️ Apaga função
    if (isset($_POST['apagar_role']) && !empty($_POST['role_para_apagar'])) {
        $role_to_delete = sanitize_text_field($_POST['role_para_apagar']);
        if (get_role($role_to_delete)) {
            remove_role($role_to_delete);
            echo '<div class="notice notice-success"><p>Função apagada com sucesso!</p></div>';
        }
    }

    $roles = wp_roles()->get_names();
    ?>
    <div class="wrap">
        <h1>Controle de Acesso às Páginas</h1>
        <h2>Funções Existentes</h2>
        <ul>
            <?php foreach ($roles as $role_key => $role_name): ?>
                <li><?php echo esc_html($role_name); ?> (<?php echo esc_html($role_key); ?>)</li>
            <?php endforeach; ?>
        </ul>

        <h2>Criar Nova Função</h2>
        <form method="post">
            <input type="text" name="nova_role" placeholder="Nome da função (ex: editor_saude)" required>
            <button type="submit" class="button button-primary">Criar Função</button>
        </form>

        <h2>Apagar Função</h2>
        <form method="post">
            <select name="role_para_apagar">
                <?php foreach ($roles as $role_key => $role_name): ?>
                    <option value="<?php echo esc_attr($role_key); ?>"><?php echo esc_html($role_name); ?></option>
                <?php endforeach; ?>
            </select>
            <button type="submit" name="apagar_role" class="button button-secondary">Apagar Função</button>
        </form>
    </div>
    <?php
}

// ✅ Metabox para selecionar funções permitidas em cada página
add_action('add_meta_boxes', function () {
    add_meta_box(
        'controle_acesso_roles',
        'Controle de Acesso (Funções)',
        'controle_acesso_roles_metabox',
        'page',
        'side'
    );
});

function controle_acesso_roles_metabox($post) {
    $roles = wp_roles()->get_names();
    $roles_selecionadas = get_post_meta($post->ID, '_roles_permitidas', true) ?: [];
    ?>
    <p>Selecione as funções que podem acessar esta página:</p>
    <?php foreach ($roles as $role_key => $role_name): ?>
        <label>
            <input type="checkbox" name="roles_permitidas[]" value="<?php echo esc_attr($role_key); ?>"
                <?php checked(in_array($role_key, $roles_selecionadas)); ?>>
            <?php echo esc_html($role_name); ?>
        </label><br>
    <?php endforeach; ?>
    <?php
}

add_action('save_post', function ($post_id) {
    if (isset($_POST['roles_permitidas'])) {
        update_post_meta($post_id, '_roles_permitidas', array_map('sanitize_text_field', $_POST['roles_permitidas']));
    } else {
        delete_post_meta($post_id, '_roles_permitidas');
    }
});

// ✅ Restringe acesso às páginas no admin
add_action('pre_get_posts', function ($query) {
    if (!is_admin() || !$query->is_main_query() || $query->get('post_type') !== 'page') {
        return;
    }

    $user = wp_get_current_user();
    if (in_array('administrator', $user->roles)) {
        return;
    }

    $allowed_ids = [];
    $pages = get_pages(['post_type' => 'page']);
    foreach ($pages as $page) {
        $roles_permitidas = get_post_meta($page->ID, '_roles_permitidas', true) ?: [];
        if (array_intersect($user->roles, $roles_permitidas)) {
            $allowed_ids[] = $page->ID;
        }
    }

    if (!empty($allowed_ids)) {
        $query->set('post__in', $allowed_ids);
    } else {
        $query->set('post__in', [0]);
    }
});

// ✅ Bloqueia edição e leitura se a função não tiver permissão
add_filter('user_has_cap', function ($allcaps, $cap, $args, $user) {
    if (in_array('administrator', $user->roles)) {
        return $allcaps;
    }

    $pagina_alvo = isset($args[2]) ? (int)$args[2] : 0;
    if (!$pagina_alvo) {
        return $allcaps;
    }

    $roles_permitidas = get_post_meta($pagina_alvo, '_roles_permitidas', true) ?: [];
    if (array_intersect($user->roles, $roles_permitidas)) {
        $allcaps['edit_page'] = true;
        $allcaps['read_page'] = true;
    } else {
        $allcaps['edit_page'] = false;
        $allcaps['read_page'] = false;
    }

    return $allcaps;
}, 10, 4);
